unit IWDynGrid;

interface

uses
  Classes, SysUtils,
  {$IFDEF VSNET}
  System.ComponentModel, System.Drawing,
  {$ENDIF}
  {$IFDEF CLR}
  IWNetComponent, IWNetClasses, IWNETBaseControl, Graphics,
  {$ELSE}
    {$IFDEF Linux}
    QControls, QGraphics,
    {$ELSE}
    Controls, Graphics,
    {$ENDIF}
  {$ENDIF}
  IWControl, IWGrids, IWFont, IWTypes, IWClientSideDatasetBase, IWForm, IWServer, swsystem,
  IWHTMLTag, IWGridCommon, IWColor, IWRenderContext;

type
  TIWCustomDynGrid = class;

  TIWDynGridBorderColors = class(TIWGridBorderColors)
  public
    procedure Assign(ASource: TPersistent); override;
  end;

  TIWDefaultDynGridCell = class(TPersistent)
  private
    FBGColor: TIWColor;
    FColor: TIWColor;
    //FFont: TIWFont;
    FOwner: TIWCustomDynGrid;
    FBorderSize: Integer;
    FBorderColors: TIWDynGridBorderColors;

    procedure SetBGColor(const Value: TIWColor);
    procedure SetColor(const Value: TIWColor);
    //procedure SetFont(const Value: TIWFont);
    function GetCellView: string;
    function GetInternalView: string;
  public
    {$IFDEF VSNET}
    constructor Create;
    {$ELSE}
    constructor Create(AOwner: TIWCustomDynGrid);
    {$ENDIF}
    destructor Destroy; override;

    procedure Assign(ASource: TPersistent); override;

    property CellView: string read GetCellView;
    property InternalView: string read GetInternalView;
  published
    property BorderColors: TIWDynGridBorderColors read FBorderColors write FBorderColors;
    property Color: TIWColor read FColor write SetColor;
    property BorderSize: Integer read FBorderSize write FBorderSize;
    property BGColor: TIWColor read FBGColor write SetBGColor;
    //property Font: TIWFont read FFont write SetFont;
  end;

  TIWCustomDynGrid = class(TIWCustomControl)
  private
    FBorderColors: TIWDynGridBorderColors;
    FBGColor: TIWColor;
    FData: TIWClientSideDatasetBase;
    FDataCellProperties: TIWDefaultDynGridCell;
    FFilterCellProperties: TIWDefaultDynGridCell;
    FTitleCellProperties: TIWDefaultDynGridCell;
    FBorderSize: Integer;
    FWrapTitles: boolean;
    procedure SetBGColor(const Value: TIWColor);
    procedure SetData(const Value: TIWClientSideDatasetBase);
    procedure SetBorderColors(const Value: TIWDynGridBorderColors);
    procedure SetDataCellProperties(const Value: TIWDefaultDynGridCell);
    procedure SetFilterCellProperties(
      const Value: TIWDefaultDynGridCell);
    procedure SetTitelCellProperties(const Value: TIWDefaultDynGridCell);
    { Private declarations }
  protected
    { Protected declarations }
    {$IFDEF VSNET}
    procedure Notification(AComponent: TPlatformComponent; AOperation: TOperation); override;
    {$ELSE}
    procedure Notification(AComponent: TComponent; AOperation: TOperation); override;
    {$ENDIF}
    procedure InitControl; override;
  public
    { Public declarations }
    destructor Destroy; override;

    function RenderHTML(AContext: TIWBaseHTMLComponentContext): TIWHTMLTag; override;

    property BorderColors: TIWDynGridBorderColors read FBorderColors write SetBorderColors;
    property BGColor: TIWColor read FBGColor write SetBGColor;
    property TitleCell: TIWDefaultDynGridCell read FTitleCellProperties write SetTitelCellProperties;
    property FilterCell: TIWDefaultDynGridCell read FFilterCellProperties write SetFilterCellProperties;
    property DataCell: TIWDefaultDynGridCell read FDataCellProperties write SetDataCellProperties;
    property Data: TIWClientSideDatasetBase read FData write SetData;
    property BorderSize: Integer read FBorderSize write FBorderSize;
    property WrapTitles: boolean read FWrapTitles write FWrapTitles;
  end;

  {$IFDEF VSNET}
  {$R icons\Atozed.Intraweb.TIWDYNGRID.bmp}
  TIWDynGrid = class;
  [ToolboxItem(true), ToolboxBitmap(typeof(TIWDynGrid), 'TIWDynGrid.bmp'),
       TIWToolPalette('Intraweb ClientSide Controls')]
  {$ENDIF}
  TIWDynGrid = class(TIWCustomDynGrid)
  published
    property BorderColors;
    property BGColor;
    property TitleCell;
    property FilterCell;
    property DataCell;
    property Data;
    property BorderSize;
    property FriendlyName;
    property Height;
    property Left;
    property Width;
    property Top;
    property Visible;
    property WrapTitles;
  end;

implementation

uses
  IWGlobal, IWResourceStrings, IWKlooch,
  IWServerControllerBase, IWAppForm, IWApplication;

function AddStyleDefinition(AStyle, ANewStyle: string): string;
begin
  if Length(AStyle) > 0 then
    result := AStyle + '; ' + ANewStyle
  else
    result := ANewStyle;
end;

{ TIWDefaultDynGridCell }

procedure TIWDefaultDynGridCell.Assign(ASource: TPersistent);
begin
  if ASource is TIWDefaultDynGridCell then begin
    //FFont.Assign((ASource as TIWDefaultDynGridCell).FFont);
    FBGColor := (ASource as TIWDefaultDynGridCell).FBGColor;
    FColor := (ASource as TIWDefaultDynGridCell).FColor;
    FOwner := (ASource as TIWDefaultDynGridCell).FOwner;
    FBorderColors.Assign((ASource as TIWDefaultDynGridCell).FBorderColors);
    FBorderSize := (ASource as TIWDefaultDynGridCell).FBorderSize;
  end
  else
    inherited Assign(ASource);
end;

{$IFDEF VSNET}
constructor TIWDefaultDynGridCell.Create;
begin
  inherited Create;
  BGColor := fromTColor(clNone);
  BorderColors := TIWDynGridBorderColors.Create;
  Color := fromTColor(clNone);
end;
{$ELSE}
constructor TIWDefaultDynGridCell.Create(AOwner: TIWCustomDynGrid);
begin
  inherited Create;
  FOwner := AOwner;
  BGColor := fromTColor(clNone);
  BorderColors := TIWDynGridBorderColors.Create;
  Color := fromTColor(clNone);
end;
{$ENDIF}

destructor TIWDefaultDynGridCell.Destroy;
begin
  BorderColors.Free;
  //FFont.Free;
  inherited Destroy;
end;

procedure TIWDefaultDynGridCell.SetBGColor(const Value: TIWColor);
begin
  FBGColor := Value;
end;

procedure TIWDefaultDynGridCell.SetColor(const Value: TIWColor);
begin
  FColor := Value;
end;

{procedure TIWDefaultDynGridCell.SetFont(const Value: TIWFont);
begin
  FFont := Value;
end;}

procedure TIWCustomDynGrid.SetBorderColors(
  const Value: TIWDynGridBorderColors);
begin
  FBorderColors.Assign(Value);
end;

function TIWDefaultDynGridCell.GetCellView: string;
begin
  Result := '<TABLE'
    + iif(BorderSize > 0, ' Border=' + IntToStr(BorderSize))
    + HTMLColor(BGColor, 'BGCOLOR')
    + iif(BorderSize > 0, HTMLColor(BorderColors.Color, 'BORDERCOLOR')
      + HTMLColor(BorderColors.Light, 'BORDERCOLORLIGHT')
      + HTMLColor(BorderColors.Dark, 'BORDERCOLORDARK'))
    + ' CELLPADDING="0"'
    + ' CELLSPACING="0"'
    + ' WIDTH="100%"'
    + ' style="border-collapse: collapse">';
end;

function TIWDefaultDynGridCell.GetInternalView: string;
begin
  result := '>';
end;

{ TIWCustomDynGrid }

procedure TIWCustomDynGrid.InitControl;
begin
  inherited;
  Width := 200;
  Height := 150;
  if not GLicense.CanIDoThis(laDynGrid) then begin
    if (GLicense.LicenseVal = ltPackagedPage) then begin
      if GServerType = stPage then begin
        raise Exception.Create('Dynamic Grids are not available in this edition of IntraWeb');
      end;
    end else begin
      raise Exception.Create('Dynamic Grids are not available in this edition of IntraWeb');
    end;
  end;
  {$IFDEF VSNET}
  FTitleCellProperties := TIWDefaultDynGridCell.Create;
  FDataCellProperties := TIWDefaultDynGridCell.Create;
  FFilterCellProperties := TIWDefaultDynGridCell.Create;
  {$ELSE}
  FTitleCellProperties := TIWDefaultDynGridCell.Create(Self);
  FDataCellProperties := TIWDefaultDynGridCell.Create(Self);
  FFilterCellProperties := TIWDefaultDynGridCell.Create(Self);
  {$ENDIF}
  FBorderColors := TIWDynGridBorderColors.Create;

  set_RenderSize(true);

  FBGColor := fromTColor(clNone);
  FWrapTitles := false;
end;

destructor TIWCustomDynGrid.Destroy;
begin
  FTitleCellProperties.Free;
  FDataCellProperties.Free;
  FFilterCellProperties.Free;
  FBorderColors.Free;
  inherited Destroy;
end;

procedure TIWCustomDynGrid.Notification(AComponent:
  {$IFDEF VSNET}TPlatformComponent{$ELSE}TComponent{$ENDIF};
  AOperation: TOperation);
begin
  inherited Notification(AComponent, AOperation);
  if AOperation = opRemove then begin
    if AComponent = FData then begin
      FData := nil;
    end;
  end;
end;

procedure TIWCustomDynGrid.SetBGColor(const Value: TIWColor);
begin
  FBGColor := Value;
end;

procedure TIWCustomDynGrid.SetData(const Value: TIWClientSideDatasetBase);
begin
  FData := Value;
  {$IFNDEF VSNET}
  if Assigned(FData) then begin
    FData.FreeNotification(Self);
  end;
  {$ENDIF}
end;

procedure TIWCustomDynGrid.SetDataCellProperties(
  const Value: TIWDefaultDynGridCell);
begin
  FDataCellProperties := Value;
end;

procedure TIWCustomDynGrid.SetFilterCellProperties(
  const Value: TIWDefaultDynGridCell);
begin
  FFilterCellProperties := Value;
end;

procedure TIWCustomDynGrid.SetTitelCellProperties(
  const Value: TIWDefaultDynGridCell);
begin
  FTitleCellProperties := Value;
end;

function TIWCustomDynGrid.RenderHTML(AContext: TIWBaseHTMLComponentContext): TIWHTMLTag;
var
  S: string;
  LScript: string;
begin
  if Assigned(FData) then begin
    S := '<TABLE'
      + iif(BorderSize > 0, ' Border=' + IntToStr(BorderSize))
      + HTMLColor(BGColor, 'BGCOLOR')
      + iif(BorderSize > 0, HTMLColor(BorderColors.Color, 'BORDERCOLOR')
        + HTMLColor(BorderColors.Light, 'BORDERCOLORLIGHT')
        + HTMLColor(BorderColors.Dark, 'BORDERCOLORDARK'))
      + ' CELLPADDING="0"'
      + ' CELLSPACING="0"'
      + ' WIDTH="100%"'
      + ' style="border-collapse: collapse">';

    LScript := {'var ' + HTMLName + '_TitleCaptions = ' + FData.RenderColNames + ';' + EOL +
      FData.RenderData(HTMLName + '_CellValues') + EOL +}
      'var ' + HTMLName + '_TableLook = ''' + S + ''';' + EOL +
      'var ' + HTMLName + '_DataLook = ''' + FDataCellProperties.CellView + ''';' + EOL +
      'var ' + HTMLName + '_TitleLook = ''' + FTitleCellProperties.CellView + ''';' + EOL +
      'var ' + HTMLName + '_TitleView = ''' + FTitleCellProperties.InternalView + ''';' + EOL +
      'var ' + HTMLName + '_FilterLook = ''' + FFilterCellProperties.CellView + ''';' + EOL +
//      'var ' + HTMLName + '_FilterLook = ''' + FFilterCellProperties.CellView + ''';' + EOL +
      'var ' + HTMLName + ';';


    with TIWComponent40Context(AContext) do begin
      AddToJavaScriptOnce(LScript);
      CacheImage(Self.HTMLName + '_ImgUp', WebApplication.InternalURLBase + '/gfx/SB_VUP2.gif');
      CacheImage(Self.HTMLName + '_ImgDown', WebApplication.InternalURLBase + '/gfx/SB_VDOWN2.gif');
      CacheImage(Self.HTMLName + '_ImgBlank', WebApplication.InternalURLBase + '/gfx/Tp.gif');
      AddScriptFile('/js/IWDynGridScript.js');
      AddToInitProc(HTMLName + ' = new DefineDynGrid("' + HTMLName + '", "' + HTMLName + '", ' +
        HTMLName + '_TableLook, ' + HTMLName + '_DataLook, ' + HTMLName + '_TitleLook, ' +
        HTMLName + '_FilterLook, CSDSFieldNames_' + FData.HTMLName + ', CSDSDataArray_' + FData.HTMLName +
        ', ' + iif(FWrapTitles, '""', '"nowrap"') + ')' + EOL +
        HTMLName + '.ImgUp = GImageCache_' + HTMLName + '_ImgUp;' + EOL + // "' + WebApplication.InternalURLBase + '/gfx/SB_VUP2.gif";' + EOL +
        HTMLName + '.ImgDown = GImageCache_' + HTMLName + '_ImgDown;' + EOL +// "' + WebApplication.InternalURLBase + '/gfx/SB_VDOWN2.gif";' + EOL +
        HTMLName + '.ImgBlank = GImageCache_' + HTMLName + '_ImgBlank;' + EOL + //"' + WebApplication.InternalURLBase + '/gfx/Tp.gif";' + EOL +
        HTMLName + '.InitTable();' + EOL + HTMLName + '.RenderTable();');
    end;
  end;
  Result := TIWHTMLTag.CreateTag('DIV'); try
    Result.AddStringParam('NAME', HTMLName);
    Result.Contents.AddText('&nbsp;');
  except FreeAndNil(Result); raise; end;
end;

{ TIWDynGridBorderColors }

procedure TIWDynGridBorderColors.Assign(ASource: TPersistent);
begin
  if ASource is TIWDynGridBorderColors then begin
    Color := (ASource as TIWDynGridBorderColors).Color;
    Light := (ASource as TIWDynGridBorderColors).Light;
    Dark := (ASource as TIWDynGridBorderColors).Dark;
  end
  else
    inherited Assign(ASource);
end;

initialization
  TIWServer.AddInternalFile('IW_JS_IWDynGridScript', '/js/IWDynGridScript.js');
  TIWServer.AddInternalFile('IW_GFX_SB_vdown2', '/gfx/SB_VDOWN2.gif');
  TIWServer.AddInternalFile('IW_GFX_SB_vup2', '/gfx/SB_VUP2.gif');
end.

